//OHSAT GAMES TUTORIAL: MEGALAGA 8 BONUS: A NEW SURPRISE!
//There's a reference to Rolling in the OHSAT Tutorial for this lesson.
//It is your duty to make the reference to Weird Al and not Fred Durst or something equally terrible.   

//https://www.ohsat.com/tutorial/#mega-drive-tutorials 

#include <genesis.h>
#include <resources.h>

#define LEFT_EDGE 0
#define RIGHT_EDGE 320
#define BOTTOM_EDGE 224

#define MAX_ENEMIES 6
#define MAX_FLY_ENEMIES 3
#define MAX_BULLETS 3
#define MAX_ENEMY_BULLETS 5
#define FLY_SPAWN_DELAY 180 // approx 3 seconds @ 60 FPS

#define ANIM_STRAIGHT 0 //ship movement
#define ANIM_MOVE 1 //ship movement

#define ANIM_WALK 0 //nemo walk
#define ANIM_DIE 7 //nemo sprite

typedef struct {
    int x, y, w, h;
    int velx, vely;
    int health;
    Sprite* sprite;
    char name[8];
} Entity;

// Globals
int offset = 0;
u16 enemiesLeft = 0;
int score = 0;
char hud_string[40];
bool flyEnemiesSpawned = FALSE;
u16 flySpawnTimer = 0;
Entity boss = {0};
int bossHealth = 20;
bool bossActive = FALSE;
bool bossPhaseTwo = FALSE;
bool bossNearDeath = FALSE;
bool bossDying = FALSE;
s16 bossFireCooldown = 0;
int bossDeathAnimStep = 0;

Entity player = {0}, player_2 = {0};
Entity enemies_top[MAX_ENEMIES];
Entity enemies_bottom[MAX_FLY_ENEMIES];
Entity bullets_p1[MAX_BULLETS], bullets_p2[MAX_BULLETS];
Entity enemyBulletsTop[MAX_ENEMY_BULLETS], enemyBulletsFly[MAX_ENEMY_BULLETS];

u16 bulletsOnScreenP1 = 0, bulletsOnScreenP2 = 0;
u16 enemyBulletsOnScreenTop = 0, enemyBulletsOnScreenFly = 0;

u16 enemyFireCooldownTop = 0;
u16 enemyFireCooldownFly = 0;

// Function declarations
void killEntity(Entity* e);
void reviveEntity(Entity* e);
void positionPlayers();
void positionEnemies();
void positionBullets();
void positionEnemyBullets();
void shootBullet(Entity* shooter, Entity* bulletArray, u16* bulletCount, int velY);
void enemyFire();
void handleCollisions();
void updateScoreDisplay();
void spawnFlyEnemies();
void spawnBoss();
void positionBoss();
void bossFire();
void handleBossDamage();
void myJoyHandler(u16 joy, u16 changed, u16 state);

int main() {
    JOY_init();
    JOY_setEventHandler(&myJoyHandler);

    SYS_disableInts();
    VDP_loadTileSet(background.tileset, 1, DMA);
    PAL_setPalette(PAL1, background.palette->data, DMA);
    PAL_setPalette(PAL2, background.palette->data, DMA);
    VDP_setScrollingMode(HSCROLL_PLANE, VSCROLL_PLANE);
    PAL_setColor(34, RGB24_TO_VDPCOLOR(0x0078f8));
    SYS_enableInts();

    SPR_init();

    // Init players
    player = (Entity){152, 192, 16, 16, 0, 0, 1, SPR_addSprite(&ship, 152, 192, TILE_ATTR(PAL1, 0, FALSE, FALSE)), "P1"};
    player_2 = (Entity){100, 192, 16, 16, 0, 0, 1, SPR_addSprite(&ship1, 100, 192, TILE_ATTR(PAL1, 0, FALSE, FALSE)), "P2"};

    // Fill background with stars
    for (int i = 0; i < 1280; i++) {
        int thex = i % 40;
        int they = i / 40;
        int val = (random() % 10) + 1;
        if (val > 3) val = 1;
        VDP_setTileMapXY(BG_B, TILE_ATTR_FULL(PAL1, 0, 0, 0, val), thex, they);
    }

    // Initialize top enemies
    enemiesLeft = 0;
    for (int i = 0; i < MAX_ENEMIES; i++) {
        Sprite* spriteTop = SPR_addSprite(&ship, i * 48, 32, TILE_ATTR(PAL2, 0, TRUE, FALSE));
        if (spriteTop) {
            enemies_top[i] = (Entity){i * 48, 32, 16, 16, 1, 0, 1, spriteTop, ""};
            sprintf(enemies_top[i].name, "E1_%d", i);
            enemiesLeft++;
        } else {
            enemies_top[i].health = 0;
            enemies_top[i].sprite = NULL;
            VDP_drawText("Top sprite failed", 2, 12 + i);
        }
    }

    // Initialize fly enemies inactive (no sprite yet)
    for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
        enemies_bottom[i].health = 0;
        enemies_bottom[i].sprite = NULL;
    }

    // Initialize bullets (off-screen, hidden)
    for (int i = 0; i < MAX_BULLETS; i++) {
        bullets_p1[i] = (Entity){0, -10, 8, 8, 0, 0, 0, SPR_addSprite(&bullet, 0, -10, TILE_ATTR(PAL1, 0, FALSE, FALSE)), ""};
        bullets_p2[i] = (Entity){0, -10, 8, 8, 0, 0, 0, SPR_addSprite(&bullet1, 0, -10, TILE_ATTR(PAL1, 0, FALSE, FALSE)), ""};
        killEntity(&bullets_p1[i]);
        killEntity(&bullets_p2[i]);
    }

    // Initialize enemy bullets (top enemies, bullet sprite)
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        enemyBulletsTop[i] = (Entity){0, 0, 8, 8, 0, 0, 0, SPR_addSprite(&bullet, 0, -10, TILE_ATTR(PAL2, 0, FALSE, FALSE)), ""};
        killEntity(&enemyBulletsTop[i]);
    }

    // Initialize enemy bullets (fly enemies, bullet1 sprite)
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        enemyBulletsFly[i] = (Entity){0, 0, 8, 8, 0, 0, 0, SPR_addSprite(&bullet1, 0, -10, TILE_ATTR(PAL2, 0, FALSE, FALSE)), ""};
        killEntity(&enemyBulletsFly[i]);
    }

    updateScoreDisplay();

    while (1) {
        VDP_setVerticalScroll(BG_B, offset -= 2);
        if (offset <= -256) offset = 0;

        positionPlayers();
        positionEnemies();
        positionBullets();
        positionEnemyBullets();
        enemyFire();
        handleCollisions();

        if (bossActive) {
            positionBoss();
        }

        SPR_update();
        SYS_doVBlankProcess();
    }

    return 0;
}

// Input Handler
void myJoyHandler(u16 joy, u16 changed, u16 state) {
    Entity* p = (joy == JOY_1) ? &player : &player_2;
    Entity* bullets = (joy == JOY_1) ? bullets_p1 : bullets_p2;
    u16* bulletCount = (joy == JOY_1) ? &bulletsOnScreenP1 : &bulletsOnScreenP2;

    if (state & BUTTON_RIGHT) {
        p->velx = 2;
        SPR_setAnim(p->sprite, ANIM_MOVE);
        SPR_setHFlip(p->sprite, TRUE);
    } else if (state & BUTTON_LEFT) {
        p->velx = -2;
        SPR_setAnim(p->sprite, ANIM_MOVE);
        SPR_setHFlip(p->sprite, FALSE);
    } else if ((changed & (BUTTON_LEFT | BUTTON_RIGHT))) {
        p->velx = 0;
        SPR_setAnim(p->sprite, ANIM_STRAIGHT);
    }

    if ((state & BUTTON_B) && (changed & BUTTON_B)) {
        shootBullet(p, bullets, bulletCount, -3);
    }
}

// Movement and position updates
void positionPlayers() {
    Entity* players[2] = {&player, &player_2};
    for (int i = 0; i < 2; i++) {
        Entity* p = players[i];
        p->x += p->velx;

        if (p->x < LEFT_EDGE) p->x = LEFT_EDGE;
        if (p->x + p->w > RIGHT_EDGE) p->x = RIGHT_EDGE - p->w;

        SPR_setPosition(p->sprite, p->x, p->y);
    }
}

void positionEnemies() {
    // Top enemies
    for (int i = 0; i < MAX_ENEMIES; i++) {
        Entity* e = &enemies_top[i];
        if (e->health > 0) {
            e->x += e->velx;
            if (e->x < LEFT_EDGE || (e->x + e->w) > RIGHT_EDGE) e->velx = -e->velx;
            SPR_setPosition(e->sprite, e->x, e->y);
        }
    }
    // Flying enemies
    for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
        Entity* f = &enemies_bottom[i];
        if (f->health > 0) {
            f->x += f->velx;

            if (f->x < LEFT_EDGE) {
                f->x = LEFT_EDGE;
                f->velx = -f->velx;
            } else if (f->x + f->w > RIGHT_EDGE) {
                f->x = RIGHT_EDGE - f->w;
                f->velx = -f->velx;
            }
            SPR_setPosition(f->sprite, f->x, f->y);
        }
    }
}

void shootBullet(Entity* shooter, Entity* bulletArray, u16* bulletCount, int velY) {
    if (*bulletCount < MAX_BULLETS) {
        for (int i = 0; i < MAX_BULLETS; i++) {
            Entity* b = &bulletArray[i];
            if (b->health == 0) {
                b->x = shooter->x + (shooter->w / 2) - (b->w / 2);
                b->y = shooter->y + (velY < 0 ? -b->h : shooter->h); // spawn above if going up, below if going down
                b->vely = velY;
                reviveEntity(b);
                SPR_setPosition(b->sprite, b->x, b->y);
                (*bulletCount)++;
                break;
            }
        }
    }
}

void positionBullets() {
    for (int i = 0; i < MAX_BULLETS; i++) {
        // Player 1 bullets
        Entity* b1 = &bullets_p1[i];
        if (b1->health > 0) {
            b1->y += b1->vely;
            if (b1->y + b1->h < 0) {
                killEntity(b1);
                if (bulletsOnScreenP1 > 0) bulletsOnScreenP1--;
            } else {
                SPR_setPosition(b1->sprite, b1->x, b1->y);
            }
        }

        // Player 2 bullets
        Entity* b2 = &bullets_p2[i];
        if (b2->health > 0) {
            b2->y += b2->vely;
            if (b2->y + b2->h < 0) {
                killEntity(b2);
                if (bulletsOnScreenP2 > 0) bulletsOnScreenP2--;
            } else {
                SPR_setPosition(b2->sprite, b2->x, b2->y);
            }
        }
    }
}

void positionEnemyBullets() {
    // Enemy bullets from top enemies (including boss)
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        Entity* eb = &enemyBulletsTop[i];
        if (eb->health > 0) {
            eb->x += eb->velx;  // <-- newly added
            eb->y += eb->vely;
            if (eb->y > BOTTOM_EDGE) {
                killEntity(eb);
                if (enemyBulletsOnScreenTop > 0) enemyBulletsOnScreenTop--;
            } else {
                SPR_setVFlip(eb->sprite, TRUE); // Ensure vertical flip
                SPR_setPosition(eb->sprite, eb->x, eb->y);
            }
        }
    }

    // Enemy bullets from fly enemies
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        Entity* eb = &enemyBulletsFly[i];
        if (eb->health > 0) {
            eb->y += eb->vely;
            if (eb->y > BOTTOM_EDGE) {
                killEntity(eb);
                if (enemyBulletsOnScreenFly > 0) enemyBulletsOnScreenFly--;
            } else {
                SPR_setVFlip(eb->sprite, TRUE); // Ensure vertical 
                eb->x += eb->velx;
                SPR_setPosition(eb->sprite, eb->x, eb->y);
            }
        }
    }
}

void enemyFire() {
    // Cooldown timers decrement
    if (enemyFireCooldownTop > 0) enemyFireCooldownTop--;
    if (enemyFireCooldownFly > 0) enemyFireCooldownFly--;

    // Fire from top enemies
    if (enemyFireCooldownTop == 0 && enemiesLeft > 0) {
        for (int i = 0; i < MAX_ENEMIES; i++) {
            Entity* e = &enemies_top[i];
            if (e->health > 0 && enemyBulletsOnScreenTop < MAX_ENEMY_BULLETS) {
                if (random() % 20 == 0) { // 1/20 chance to fire
                    for (int j = 0; j < MAX_ENEMY_BULLETS; j++) {
                        Entity* b = &enemyBulletsTop[j];
                        if (b->health == 0) {
                            b->x = e->x + (e->w / 2) - (b->w / 2);
                            b->y = e->y + e->h;
                            b->vely = 2;
                            reviveEntity(b);
                            enemyBulletsOnScreenTop++;
                            enemyFireCooldownTop = 30;
                            break;
                        }
                    }
                    break;
                }
            }
        }
    }

    // Spawn fly enemies after top enemies all die
    if (enemiesLeft == 0) {
        flySpawnTimer++;
        if (!flyEnemiesSpawned && flySpawnTimer > FLY_SPAWN_DELAY) {
            spawnFlyEnemies();
            flyEnemiesSpawned = TRUE;
        }
    }

    // Fire from fly enemies
    if (flyEnemiesSpawned && enemyFireCooldownFly == 0) {
        for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
            Entity* f = &enemies_bottom[i];
            if (f->health > 0 && enemyBulletsOnScreenFly < MAX_ENEMY_BULLETS) {
                if (random() % 20 == 0) {
                    for (int j = 0; j < MAX_ENEMY_BULLETS; j++) {
                        Entity* b = &enemyBulletsFly[j];
                        if (b->health == 0) {
                            b->x = f->x + (f->w / 2) - (b->w / 2);
                            b->y = f->y + f->h;
                            b->vely = 2;
                            reviveEntity(b);
                            enemyBulletsOnScreenFly++;
                            enemyFireCooldownFly = 30;
                            break;
                        }
                    }
                    break;
                }
            }
        }
    }

    // Boss appears after all fly enemies are defeated
    if (flyEnemiesSpawned) {
        bool allDead = TRUE;
        for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
            if (enemies_bottom[i].health > 0) {
                allDead = FALSE;
                break;
            }
        }
        if (allDead && !bossActive) {
            spawnBoss();
            bossActive = TRUE;
        }
    }

    // Boss fire logic
    if (bossActive && !bossDying) {
        if (bossFireCooldown <= 0) {
            // In phase 2, fire more shots per cooldown (burst)
            int shotsToFire = bossPhaseTwo ? 3 : 1;  // 3 shots per fire in phase 2, else 1 shot
            
            for (int i = 0; i < shotsToFire; i++) {
                bossFire();
                // Optional: add small delay or variation between shots if your engine supports it
            }
            
            // Set a shorter cooldown in phase 2 for faster firing rate
            bossFireCooldown = bossPhaseTwo ? 15 : 30;  
        } else {
            bossFireCooldown--;
            if (bossFireCooldown < 0) bossFireCooldown = 0;
        }
    }
}

void handleCollisions() {
    // Player bullets vs top enemies
    for (int i = 0; i < MAX_BULLETS; i++) {
        Entity* b1 = &bullets_p1[i];
        if (b1->health > 0) {
            for (int j = 0; j < MAX_ENEMIES; j++) {
                Entity* e = &enemies_top[j];
                if (e->health > 0) {
                    if (b1->x < e->x + e->w && b1->x + b1->w > e->x &&
                        b1->y < e->y + e->h && b1->y + b1->h > e->y) {
                        killEntity(b1);
                        bulletsOnScreenP1--;
                        e->health--;
                        if (e->health <= 0) {
                            killEntity(e);
                            enemiesLeft--;
                            score += 10;
                            updateScoreDisplay();
                        }
                        break;
                    }
                }
            }
        }
        Entity* b2 = &bullets_p2[i];
        if (b2->health > 0) {
            for (int j = 0; j < MAX_ENEMIES; j++) {
                Entity* e = &enemies_top[j];
                if (e->health > 0) {
                    if (b2->x < e->x + e->w && b2->x + b2->w > e->x &&
                        b2->y < e->y + e->h && b2->y + b2->h > e->y) {
                        killEntity(b2);
                        bulletsOnScreenP2--;
                        e->health--;
                        if (e->health <= 0) {
                            killEntity(e);
                            enemiesLeft--;
                            score += 10;
                            updateScoreDisplay();
                        }
                        break;
                    }
                }
            }
        }
    }

    // Player bullets vs fly enemies
    if (flyEnemiesSpawned) {
        for (int i = 0; i < MAX_BULLETS; i++) {
            Entity* b1 = &bullets_p1[i];
            if (b1->health > 0) {
                for (int j = 0; j < MAX_FLY_ENEMIES; j++) {
                    Entity* f = &enemies_bottom[j];
                    if (f->health > 0) {
                        if (b1->x < f->x + f->w && b1->x + b1->w > f->x &&
                            b1->y < f->y + f->h && b1->y + b1->h > f->y) {
                            killEntity(b1);
                            bulletsOnScreenP1--;
                            f->health--;
                            if (f->health <= 0) {
                                killEntity(f);
                                score += 15;
                                updateScoreDisplay();
                            }
                            break;
                        }
                    }
                }
            }
            Entity* b2 = &bullets_p2[i];
            if (b2->health > 0) {
                for (int j = 0; j < MAX_FLY_ENEMIES; j++) {
                    Entity* f = &enemies_bottom[j];
                    if (f->health > 0) {
                        if (b2->x < f->x + f->w && b2->x + b2->w > f->x &&
                            b2->y < f->y + f->h && b2->y + b2->h > f->y) {
                            killEntity(b2);
                            bulletsOnScreenP2--;
                            f->health--;
                            if (f->health <= 0) {
                                killEntity(f);
                                score += 15;
                                updateScoreDisplay();
                            }
                            break;
                        }
                    }
                }
            }
        }
    }

    // Enemy bullets vs players
    for (int i = 0; i < MAX_ENEMY_BULLETS; i++) {
        Entity* ebTop = &enemyBulletsTop[i];
        if (ebTop->health > 0) {
            if (ebTop->x < player.x + player.w && ebTop->x + ebTop->w > player.x &&
                ebTop->y < player.y + player.h && ebTop->y + ebTop->h > player.y) {
                killEntity(ebTop);
                enemyBulletsOnScreenTop--;
                // TODO: player hit logic (lives, death, etc)
            }
            if (ebTop->x < player_2.x + player_2.w && ebTop->x + ebTop->w > player_2.x &&
                ebTop->y < player_2.y + player_2.h && ebTop->y + ebTop->h > player_2.y) {
                killEntity(ebTop);
                enemyBulletsOnScreenTop--;
                // TODO: player 2 hit logic
            }
        }
        Entity* ebFly = &enemyBulletsFly[i];
        if (ebFly->health > 0) {
            if (ebFly->x < player.x + player.w && ebFly->x + ebFly->w > player.x &&
                ebFly->y < player.y + player.h && ebFly->y + ebFly->h > player.y) {
                killEntity(ebFly);
                enemyBulletsOnScreenFly--;
                // TODO: player hit logic
            }
            if (ebFly->x < player_2.x + player_2.w && ebFly->x + ebFly->w > player_2.x &&
                ebFly->y < player_2.y + player_2.h && ebFly->y + ebFly->h > player_2.y) {
                killEntity(ebFly);
                enemyBulletsOnScreenFly--;
                // TODO: player 2 hit logic
            }
        }
    }
    //boss collision
        if (bossActive && !bossDying) {
        for (int i = 0; i < MAX_BULLETS; i++) {
            Entity* b1 = &bullets_p1[i];
            Entity* b2 = &bullets_p2[i];

            Entity* bullets[] = {b1, b2};

            for (int b = 0; b < 2; b++) {
                Entity* bullet = bullets[b];
                if (bullet->health > 0 &&
                    bullet->x < boss.x + boss.w && bullet->x + bullet->w > boss.x &&
                    bullet->y < boss.y + boss.h && bullet->y + bullet->h > boss.y) {
                    killEntity(bullet);
                    if (b == 0) bulletsOnScreenP1--;
                    else bulletsOnScreenP2--;
                    bossHealth--;

                    if (bossHealth == 10) {
                        bossPhaseTwo = TRUE;
                        boss.velx *= 2;
                    } else if (bossHealth == 5) {
                        bossNearDeath = TRUE;
                    } else if (bossHealth <= 0) {
                        bossDying = TRUE;
                        boss.velx = 0;
                        boss.y -= 10; // upward nudge
                    }
                    break;
                }
            }
        }
    }
}

void killEntity(Entity* e) {
    e->health = 0;
    SPR_setVisibility(e->sprite, HIDDEN);
}

void reviveEntity(Entity* e) {
    e->health = 1;
    SPR_setVisibility(e->sprite, VISIBLE);
}

void updateScoreDisplay() {
    sprintf(hud_string, "SCORE: %04d ENEMIES: %02d", score, enemiesLeft);
    VDP_clearTextArea(0, 0, 40, 1);
    VDP_drawText(hud_string, 0, 0);
}

void spawnFlyEnemies() {
    // Example positions (randomized or fixed, but different)
    int spawnX[MAX_FLY_ENEMIES] = {40, 120, 200};
    int spawnY[MAX_FLY_ENEMIES] = {40, 60, 80};

    for (int i = 0; i < MAX_FLY_ENEMIES; i++) {
        if (enemies_bottom[i].sprite == NULL) {
            enemies_bottom[i].sprite = SPR_addSprite(&efly, spawnX[i], spawnY[i], TILE_ATTR(PAL2, 0, TRUE, FALSE));
        }
        enemies_bottom[i].x = spawnX[i];
        enemies_bottom[i].y = spawnY[i];
        enemies_bottom[i].w = 16;
        enemies_bottom[i].h = 16;

        // Start moving right initially, velocity can be positive or negative
        enemies_bottom[i].velx = (i % 2 == 0) ? 1 : -1;
        enemies_bottom[i].vely = 0;

        enemies_bottom[i].health = 1;
        SPR_setAnim(enemies_bottom[i].sprite, 0);
        SPR_setVisibility(enemies_bottom[i].sprite, VISIBLE);
        SPR_setPosition(enemies_bottom[i].sprite, enemies_bottom[i].x, enemies_bottom[i].y);
    }
}

void spawnBoss() {
    boss.x = 128;
    boss.y = 40;
    boss.w = 32;
    boss.h = 32;
    boss.velx = 1;
    boss.vely = 0;
    boss.health = bossHealth;
    boss.sprite = SPR_addSprite(&nemo_sprite, boss.x, boss.y, TILE_ATTR(PAL2, 0, FALSE, FALSE));
    SPR_setAnim(boss.sprite, 0);
    SPR_setVisibility(boss.sprite, VISIBLE);
}

void positionBoss() {
    if (!bossDying) {
        // Normal left-right movement
        boss.x += boss.velx;
        // If boss hits RIGHT_EDGE, flip direction and sprite
        if (boss.x + boss.w > RIGHT_EDGE) {
            boss.x = RIGHT_EDGE - boss.w;  // prevent overflow
            boss.velx = -boss.velx;        // change direction
            SPR_setHFlip(boss.sprite, TRUE);  // face left
        }

        // If boss hits LEFT_EDGE, flip direction and sprite
        else if (boss.x < LEFT_EDGE) {
            boss.x = LEFT_EDGE;           // prevent overflow
            boss.velx = -boss.velx;       // change direction
            SPR_setHFlip(boss.sprite, FALSE); // face right
        }
        SPR_setPosition(boss.sprite, boss.x, boss.y);
    } else {
        // Boss dying animation sequence
        if (bossDeathAnimStep < 32) {  
            // Move boss upward slowly until 32 pixels
            boss.y -= 2;  // move 2 pixels per frame upward
            bossDeathAnimStep += 2;
        } else {
            // Then fall down off the screen
            boss.y += 4; // fall speed, can tweak
            if (boss.y > BOTTOM_EDGE) {
                // Once off screen, hide boss and deactivate
                killEntity(&boss);
                bossActive = FALSE;
                bossDying = FALSE;
                bossDeathAnimStep = 0;
            }
        }
        SPR_setPosition(boss.sprite, boss.x, boss.y);
        SPR_setAnim(boss.sprite, 7);
    }
}

void bossFire() {
    for (int i = 0; i < 3; i++) {
        Entity* b = NULL;
        for (int j = 0; j < MAX_ENEMY_BULLETS; j++) {
            if (enemyBulletsTop[j].health == 0) {
                b = &enemyBulletsTop[j];
                break;
            }
        }
        if (b == NULL) continue;

        b->x = boss.x + (boss.w / 2) - 4;
        b->y = boss.y + boss.h;
        b->vely = 2;

        // Spread bullets by setting horizontal velocity
        if (i == 0) b->velx = -1;
        else if (i == 2) b->velx = 1;
        else b->velx = 0;

        b->sprite = SPR_addSprite(bossPhaseTwo ? &bullet1 : &bullet, b->x, b->y, TILE_ATTR(PAL2, 0, FALSE, FALSE));
        reviveEntity(b);
        enemyBulletsOnScreenTop++;
    }
}



/////////////////CHANGE LOG//////////////////

/*

Added functions for boss logic and have condensed down resources.res to strip out calls
to unused assets. 

*/

////////////////////NOTES////////////////////

/*

Code Summary: Function overview

🎮 void myJoyHandler(u16 joy, u16 changed, u16 state)
Purpose:
Handles player input from the joypad.

Key Logic:

Determines which player the input is for (JOY_1 vs. others).

Updates horizontal velocity and animation state based on BUTTON_LEFT / BUTTON_RIGHT.

Resets animation and velocity when direction changes.

When BUTTON_B is pressed (and it's a new press), the player fires a bullet using shootBullet().

👣 void positionPlayers()
Purpose:
Updates the position and screen bounds of both players.

Key Logic:

Applies velocity to each player's x position.

Clamps player position to within screen boundaries.

Updates sprite position accordingly.

👾 void positionEnemies()
Purpose:
Handles movement logic for both top and flying enemies.

Key Logic:

Moves top enemies left/right, bounces them off screen edges.

Flying enemies bounce similarly but are initialized separately.

Positions are updated and reflected in the sprites.

🔫 void shootBullet(...)
Purpose:
Spawns a bullet from a player entity.

Key Logic:

Finds a free bullet slot in the bullet array.

Sets bullet position to appear centered relative to the shooter.

Sets vertical speed and marks the bullet as active (reviveEntity).

Updates on-screen bullet count.

💥 void positionBullets()
Purpose:
Moves player bullets and handles when they go off-screen.

Key Logic:

Iterates over both players’ bullets.

Moves bullets along the Y-axis using their velocity.

Destroys bullets that leave the screen, updates bullet counters.

Updates sprite positions.

💣 void positionEnemyBullets()
Purpose:
Updates enemy bullet positions and destroys them if off-screen.

Key Logic:

Top bullets and fly bullets handled in separate loops.

Bullets move downward; top bullets can also have horizontal velocity (boss).

When a bullet leaves the bottom of the screen, it’s killed and counter is decremented.

🧠 void enemyFire()
Purpose:
Controls when enemies fire bullets and manages boss/fly enemy spawning.

Key Logic:

Decrements cooldowns for top and fly enemies.

Fires randomly (1 in 20 chance) if cooldown is 0 and bullet slots are free.

Once top enemies are all dead, it triggers a timer to spawn fly enemies.

After all fly enemies are dead, the boss is spawned.

Boss fires bullets in bursts depending on phase (phase 2 = faster + spread).

🔍 void handleCollisions()
Purpose:
Handles all collision detection and reactions:

Player bullets vs. top enemies

Player bullets vs. fly enemies

Enemy bullets vs. players

Player bullets vs. boss

Key Logic:

Axis-Aligned Bounding Box (AABB) checks.

When a bullet hits an enemy:

Enemy health is decremented.

If health hits 0, kill entity and update score.

Boss-specific logic:

Boss has phases: phase two at 10 HP, near-death at 5, dies at 0.

Triggers animation and movement changes.

☠️ void killEntity(Entity* e)
Purpose:
Marks an entity as "dead" by hiding it and setting health to 0.

❤️ void reviveEntity(Entity* e)
Purpose:
Revives an entity by making it visible and restoring health to 1.

🧮 void updateScoreDisplay()
Purpose:
Updates HUD to reflect current score and remaining enemies.

Key Logic:

Draws score and enemy count to the top of the screen.

🐝 void spawnFlyEnemies()
Purpose:
Spawns flying enemies once top enemies are eliminated.

Key Logic:

Assigns predefined positions, sprites, sizes, velocities.

Alternates direction for dynamic movement.

Makes sprites visible.

🐙 void spawnBoss()
Purpose:
Spawns the boss once all fly enemies are eliminated.

Key Logic:

Initializes boss position, size, velocity.

Sets sprite and visibility.

🧍‍♂️ void positionBoss()
Purpose:
Handles boss movement and death animation.

Key Logic:

Boss moves left-right and bounces off screen edges.

During death, animates upward movement then fall-off.

Hides sprite and resets death state when off-screen.

🔥 void bossFire()
Purpose:
Boss fires a 3-way bullet spread (left, center, right) depending on phase.

Key Logic:

Spawns 3 bullets with different horizontal velocities.

Uses a different sprite in phase two.

Adds bullets to the active list if free slot found.

🧾 Summary Table
Function	Main Role
myJoyHandler	Processes player input and shooting
positionPlayers	Updates player positions and boundaries
positionEnemies	Moves top and flying enemies
shootBullet	Spawns a bullet from a player
positionBullets	Moves and cleans up player bullets
positionEnemyBullets	Moves and manages enemy bullets
enemyFire	Controls when and how enemies/boss shoot
handleCollisions	Manages all collisions and their effects
killEntity	Marks an entity as dead
reviveEntity	Revives an entity and makes it visible
updateScoreDisplay	Refreshes HUD with score/enemy count
spawnFlyEnemies	Spawns fly enemies with patterns
spawnBoss	Initializes and displays boss
positionBoss	Handles boss movement and death animation
bossFire	Fires boss bullets in spread pattern

*/

/////////EXPERIMENTATION IDEAS///////////////

/*

I've already implemented a form of SPAM protection in my bonus code. 
Therefore, let's get creative and see if we can bring NEMO back. 

*/

///////////ERROR HANDLING////////////////////
/*
Bugs

Enemy count isn't updating properly after the first wave is defeated. 
Note:Rather than further define the parameters for the enemiesLeft, I may strip this out entirely. 

The nemo_sprite needs an HFLIP when shifting from right to left. (fixed)

The ANIM_DIE animation isn't working when the boss is defeated. (fixed)

The nemo_sprite boss.sprite stops firing after entering the second phase. 
Note: I think I'm going to add a new bullet sprite unique to nemo_sprite. 

*/